/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: IOAccessDlg.cpp 12626 2018-10-30 14:55:29Z LuisContreras $:

  Description:
    Device I/O Access Dialog class

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2010-03-15  Window is only updated if last redraw is finished
    2008-11-26  Wrong data size will be now adjusted
    2007-03-29  Verify option for written outputs
    2006-06-29  initial version

**************************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
/// \file IOAccessDlg.cpp
///   Device I/O Access Dialog class
///////////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "cifXTest.h"
#include "IOAccessDlg.h"
#include "CifXDeviceBase.h"
#include "CifXTestDlg.h"
#include ".\ioaccessdlg.h"

#define CIFX_IO_TIMEOUT 0

IMPLEMENT_DYNAMIC(CIOAccessDlg, CBaseDialog)

static struct IO_DATA_UPDATERATEStag
{
  LPCTSTR       szDescription;
  unsigned long ulTimeout;

} s_atUpdateRates[] =
{
  {_T("0 ms"),    0},
  {_T("1 ms"),    1},
  {_T("2 ms"),    2},
  {_T("5 ms"),    5},
  {_T("10 ms"),  10},
  {_T("20 ms"),  20},
  {_T("50 ms"),  50},
  {_T("100 ms"), 100},
  {_T("200 ms"), 200},
  {_T("500 ms"), 500},
  {_T("1 s"),   1000},
};

/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent window
/////////////////////////////////////////////////////////////////////////////
CIOAccessDlg::CIOAccessDlg(CWnd* pParent /*=NULL*/)
	: CBaseDialog(CIOAccessDlg::IDD, pParent)
  , m_ulRecvArea(0)
  , m_ulRecvOffset(0)
  , m_ulRecvSize(0)
  , m_pabRecvBuffer(0)
  , m_ulSendArea(0)
  , m_ulSendOffset(0)
  , m_ulSendSize(0)
  , m_pabSendBuffer(NULL)
  , m_lLastReadResult(CIFX_NO_ERROR)
  , m_lLastWriteResult(CIFX_NO_ERROR)
  , m_fCyclicOutput(false)
  , m_fAutoIncrementData(false)
  , m_fDeviceValid(false)
  , m_fVerifyOutputs(false)
  , m_fSendUpdateActive(false)
  , m_fRecvUpdateActive(false)
{
  InitializeCriticalSection(&m_tcsLengthChange);
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CIOAccessDlg::~CIOAccessDlg()
{
  m_ulRecvSize = 0;
  m_ulSendSize = 0;
  
  free(m_pabRecvBuffer);
  m_pabRecvBuffer = NULL;
  free(m_pabSendBuffer);
  m_pabSendBuffer = NULL;

  DeleteCriticalSection(&m_tcsLengthChange);
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV support
///   \param pDX 
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::DoDataExchange(CDataExchange* pDX)
{
  CBaseDialog::DoDataExchange(pDX);

  DDX_Control(pDX, IDC_COMBO_INPUTAREA,  m_cInputAreaCtrl);
  DDX_Control(pDX, IDC_COMBO_OUTPUTAREA, m_cOutputAreaCtrl);
  DDX_Control(pDX, IDC_SEND_DATA,        m_cOutputDataCtrl);
  DDX_Control(pDX, IDC_RECV_DATA,        m_cInputDataCtrl);

  DDX_Control(pDX, IDC_RECV_LENGTH,      m_cInputDataLenCtrl);
  DDX_Control(pDX, IDC_RECV_OFFSET,      m_cInputDataOffsetCtrl);
  DDX_Control(pDX, IDC_SEND_LENGTH,      m_cOutputDataLenCtrl);
  DDX_Control(pDX, IDC_SEND_OFFSET,      m_cOutputDataOffsetCtrl);

  DDX_Control(pDX, IDC_COMBO_UPDATERATE, m_cUpdateRateCtrl);
  DDX_Control(pDX, IDC_CHK_CYCLIC,       m_cCyclicSendCtrl);
  DDX_Control(pDX, IDC_CHK_AUTOINCR,     m_cAutoIncrementCtrl);
  DDX_Control(pDX, IDC_CHK_VERIFYOUTPUT, m_cVerifyOutputsCtrl);
}


BEGIN_MESSAGE_MAP(CIOAccessDlg, CBaseDialog)
  ON_WM_SHOWWINDOW()
  ON_EN_KILLFOCUS(IDC_RECV_OFFSET, OnEnKillfocusRecvOffset)
  ON_EN_KILLFOCUS(IDC_RECV_LENGTH, OnEnKillfocusRecvLength)
  ON_EN_KILLFOCUS(IDC_SEND_OFFSET, OnEnKillfocusSendOffset)
  ON_EN_KILLFOCUS(IDC_SEND_LENGTH, OnEnKillfocusSendLength)
  ON_BN_CLICKED(IDC_BTN_UPDATE, OnBnClickedBtnUpdate)
  ON_EN_UPDATE(IDC_SEND_DATA, OnEnUpdateSendData)
  ON_EN_KILLFOCUS(IDC_SEND_DATA, OnEnKillfocusSendData)
  ON_CBN_SELCHANGE(IDC_COMBO_UPDATERATE, OnCbnSelchangeComboUpdaterate)
  ON_BN_CLICKED(IDC_CHK_CYCLIC, OnBnClickedChkCyclic)
  ON_BN_CLICKED(IDC_CHK_AUTOINCR, OnBnClickedChkAutoincr)
  ON_MESSAGE(WM_UPDATE_SEND, UpdateSendData)
  ON_MESSAGE(WM_UPDATE_RECV, UpdateRecvData)
  ON_BN_CLICKED(IDC_CHK_VERIFYOUTPUT, OnBnClickedChkVerifyoutput)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
/// First Time dialog initialization
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CIOAccessDlg::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  m_pabRecvBuffer = (unsigned char*)malloc(m_ulRecvSize);
  m_pabSendBuffer = (unsigned char*)malloc(m_ulSendSize);

  TCHAR szBuffer[12] = {0};
  m_cInputDataLenCtrl.SetWindowText(    _ltot( m_ulRecvSize,    szBuffer, 10));
  m_cInputDataOffsetCtrl.SetWindowText( _ltot( m_ulRecvOffset,  szBuffer, 10));

  m_cOutputDataLenCtrl.SetWindowText(   _ltot( m_ulSendSize,    szBuffer, 10));
  m_cOutputDataOffsetCtrl.SetWindowText(_ltot( m_ulSendOffset,  szBuffer, 10));
  
/*  
  m_cFixedFont.CreatePointFont(80, _T("Courier new"));
  
  GetDlgItem(IDC_SEND_DATA)->SetFont(&m_cFixedFont);
  GetDlgItem(IDC_RECV_DATA)->SetFont(&m_cFixedFont);
*/

  for(int iIdx = 0; iIdx < sizeof(s_atUpdateRates) / sizeof(s_atUpdateRates[0]); ++iIdx)
  {
    int iItem = m_cUpdateRateCtrl.AddString(s_atUpdateRates[iIdx].szDescription);
    m_cUpdateRateCtrl.SetItemData(iItem, s_atUpdateRates[iIdx].ulTimeout);  

    if(m_ulThreadTimeout == s_atUpdateRates[iIdx].ulTimeout)
      m_cUpdateRateCtrl.SetCurSel(iItem);
  }

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Cyclic I/O Exchange thread function
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::ThreadFunction(void)
{
  if( m_fDeviceValid &&
      (NULL != CcifXTestDlg::s_pcDevice) )
  {
    EnterCriticalSection(&m_tcsLengthChange);

    m_lLastReadResult = CcifXTestDlg::s_pcDevice->IORead(m_ulRecvArea,
                                                        m_ulRecvOffset,
                                                        m_ulRecvSize,
                                                        m_pabRecvBuffer,
                                                        CIFX_IO_TIMEOUT);
    if(m_fCyclicOutput)
    {
      m_lLastWriteResult = CcifXTestDlg::s_pcDevice->IOWrite(m_ulSendArea,
                                                            m_ulSendOffset,
                                                            m_ulSendSize,
                                                            m_pabSendBuffer,
                                                            CIFX_IO_TIMEOUT);

      if( (m_fVerifyOutputs) &&
          ( (CIFX_NO_ERROR == m_lLastWriteResult) ||
            (CIFX_DEV_NO_COM_FLAG == m_lLastWriteResult) ) )
      {
        /* Verify the outputs */
        long lRet = CcifXTestDlg::s_pcDevice->IOReadSend(m_ulSendArea,
                                                         m_ulSendOffset,
                                                         m_ulSendSize,
                                                         m_abCompareBuffer);

        if(CIFX_NO_ERROR != lRet)
        {
          /* Error reading send data */
          CString csTemp;
          csTemp.Format(_T("Error verifying output data. Unable to read back send data (lRet=0x%08X)!"),
                        lRet);

          AfxMessageBox(csTemp);

        } else if(0 != memcmp(m_pabSendBuffer, m_abCompareBuffer, m_ulSendSize))
        {
          /* compare reading reading back outputs */
          AfxMessageBox(_T("Compare error reading back outputs!"));

        }
      }

      /* Check if we have already drawn the last changes */
      if(!m_fSendUpdateActive)
      {
        m_fSendUpdateActive = true;
        PostMessage(WM_UPDATE_SEND); // replaces UpdateSendData();
      }
    }

    LeaveCriticalSection(&m_tcsLengthChange);

    /* Check if we have already drawn the last changes */
    if(!m_fRecvUpdateActive)
    {
      m_fRecvUpdateActive = true;
      PostMessage(WM_UPDATE_RECV); // replaces UpdateRecvData();
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Device has changed or Dialog has been activated
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  if(pcDevice != NULL)
  {
    while(m_cInputAreaCtrl.GetCount() > 0)
      m_cInputAreaCtrl.DeleteString(0);

    while(m_cOutputAreaCtrl.GetCount() > 0)
      m_cOutputAreaCtrl.DeleteString(0);

    CHANNEL_INFORMATION* ptChannelInfo = pcDevice->GetChannelInfo();
    TCHAR szBuffer[12] = {0};

    for(unsigned long ulInArea = 0; ulInArea < ptChannelInfo->ulIOInAreaCnt; ++ulInArea)
      m_cInputAreaCtrl.AddString( _ultot(ulInArea, szBuffer, 10));

    for(unsigned long ulOutArea = 0; ulOutArea < ptChannelInfo->ulIOOutAreaCnt; ++ulOutArea)
      m_cOutputAreaCtrl.AddString( _ultot(ulOutArea, szBuffer, 10));

    m_cInputAreaCtrl.SetCurSel(0);
    m_cOutputAreaCtrl.SetCurSel(0);

    EnterCriticalSection(&m_tcsLengthChange);
    m_fDeviceValid = true;
    LeaveCriticalSection(&m_tcsLengthChange);

  } else
  {
    // Make sure the thread will end device interaction, before device is really closed
    EnterCriticalSection(&m_tcsLengthChange);
    m_fDeviceValid = false;
    LeaveCriticalSection(&m_tcsLengthChange);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Input Area offset has changed
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnKillfocusRecvOffset()
{
  CString csTemp;
  GetDlgItem(IDC_RECV_OFFSET)->GetWindowText(csTemp);

  m_ulRecvOffset = (unsigned long)_ttol(csTemp);
  
}

/////////////////////////////////////////////////////////////////////////////
/// Input data length has changed
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnKillfocusRecvLength()
{
  CString csTemp;
  GetDlgItem(IDC_RECV_LENGTH)->GetWindowText(csTemp);

  EnterCriticalSection(&m_tcsLengthChange);

  m_ulRecvSize = (unsigned long)_ttol(csTemp);
  m_pabRecvBuffer = (unsigned char*)realloc(m_pabRecvBuffer, m_ulRecvSize);

  LeaveCriticalSection(&m_tcsLengthChange);
}

/////////////////////////////////////////////////////////////////////////////
/// Output data Offset has changed
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnKillfocusSendOffset()
{
  CString csTemp;
  GetDlgItem(IDC_SEND_OFFSET)->GetWindowText(csTemp);

  m_ulSendOffset = (unsigned long)_ttol(csTemp);
}

/////////////////////////////////////////////////////////////////////////////
/// Output Length has changed
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnKillfocusSendLength()
{
/*
  CString csTemp;
  GetDlgItem(IDC_SEND_LENGTH)->GetWindowText(csTemp);

  EnterCriticalSection(&m_tcsLengthChange);

  m_ulSendSize = (unsigned long)_ttol(csTemp);
  m_pabSendBuffer = (unsigned char*)realloc(m_pabSendBuffer, m_ulSendSize);

  LeaveCriticalSection(&m_tcsLengthChange);
*/
}

/////////////////////////////////////////////////////////////////////////////
/// Update Input area content
/////////////////////////////////////////////////////////////////////////////
LRESULT CIOAccessDlg::UpdateRecvData(WPARAM /*wParam = 0*/, LPARAM /*lParam = 0*/)
{
  CString csRecvData;

  EnterCriticalSection(&m_tcsLengthChange);

  if( (CIFX_NO_ERROR == m_lLastReadResult) ||
      (CIFX_DEV_NO_COM_FLAG == m_lLastReadResult) )
  {
    for(unsigned long ulByte = 0; ulByte < m_ulRecvSize; ++ulByte)
    {

      if( (ulByte > 0) && ((ulByte % 16) == 0) )
        csRecvData.Append(_T("\r\n"));

      csRecvData.AppendFormat(_T("%02X "), m_pabRecvBuffer[ulByte]);
    }
  }
 
  LeaveCriticalSection(&m_tcsLengthChange);

  m_cInputDataCtrl.SetWindowText(csRecvData);

  CString csLastError;
  csLastError.Format(_T("0x%08X\r\n"), m_lLastReadResult);
  csLastError.Append(CcifXTestDlg::s_pcDevice->GetErrorDescription(m_lLastReadResult));

  GetDlgItem(IDC_INPUT_LASTERR)->SetWindowText(csLastError);

  m_fRecvUpdateActive = false;

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Update Output area content
/////////////////////////////////////////////////////////////////////////////
LRESULT CIOAccessDlg::UpdateSendData(WPARAM /*wParam = 0*/, LPARAM /*lParam = 0*/)
{
  if( (m_fAutoIncrementData) && 
        ( (CIFX_NO_ERROR == m_lLastWriteResult) ||
          (CIFX_DEV_NO_COM_FLAG == m_lLastWriteResult) ) )
  {
    CString csOutput;

    EnterCriticalSection(&m_tcsLengthChange);
    for(unsigned long ulIdx = 0; ulIdx < m_ulSendSize; ++ulIdx)
    {
      if( (ulIdx > 0) && ((ulIdx %  16) == 0) )
        csOutput.Append(_T("\r\n"));
      
      csOutput.AppendFormat(_T("%02X "), ++m_pabSendBuffer[ulIdx]);     
    }
    m_cOutputDataCtrl.SetWindowText(csOutput);
    LeaveCriticalSection(&m_tcsLengthChange);
  }

  CString csLastError;
  csLastError.Format(_T("0x%08X\r\n"), m_lLastWriteResult);
  csLastError.Append(CcifXTestDlg::s_pcDevice->GetErrorDescription(m_lLastWriteResult));

  GetDlgItem(IDC_OUTPUT_LASTERR)->SetWindowText(csLastError);

  m_fSendUpdateActive = false;

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Insert Output data into dual port and toggle bits
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnBnClickedBtnUpdate()
{
  // Check data if eceleration buttons are used
  if( ValidateSendData())
  {
    m_lLastWriteResult = CcifXTestDlg::s_pcDevice->IOWrite(m_ulSendArea,
                                                          m_ulSendOffset,
                                                          m_ulSendSize,
                                                          m_pabSendBuffer,
                                                          CIFX_IO_TIMEOUT);
    if( (m_fVerifyOutputs) &&
        ( (CIFX_NO_ERROR == m_lLastWriteResult) ||
          (CIFX_DEV_NO_COM_FLAG == m_lLastWriteResult) ) )
    {
      /* Verify the outputs */
      long lRet = CcifXTestDlg::s_pcDevice->IOReadSend(m_ulSendArea,
                                                        m_ulSendOffset,
                                                        m_ulSendSize,
                                                        m_abCompareBuffer);

      if(CIFX_NO_ERROR != lRet)
      {
        /* Error reading send data */
        CString csTemp;
        csTemp.Format(_T("Error verifying output data. Unable to read back send data (lRet=0x%08X)!"),
                      lRet);

        AfxMessageBox(csTemp);

      } else if(0 != memcmp(m_pabSendBuffer, m_abCompareBuffer, m_ulSendSize))
      {
        /* compare reading reading back outputs */
        AfxMessageBox(_T("Compare error reading back outputs!"));

      }
    }

    UpdateSendData();
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Output data is changed by user (filter out invalid characters)
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnUpdateSendData()
{
  //TODO: Filter invalid characters
}

/////////////////////////////////////////////////////////////////////////////
/// Validate Output data
/////////////////////////////////////////////////////////////////////////////
bool CIOAccessDlg::ValidateSendData()
{
  bool fRet = false;
  
  // Check if device still exists
  if( NULL == CcifXTestDlg::s_pcDevice) return fRet;

  CString csTemp;
  m_cOutputDataCtrl.GetWindowText(csTemp);

  int iDataLen = 0;

  if(csTemp.GetLength() > 0)
  {
    // remove all whitespace characters
    csTemp.Remove(_T(' '));
    csTemp.Remove(_T('\r'));
    csTemp.Remove(_T('\n'));

    int iStrLen = csTemp.GetLength();

    if(iStrLen % 2)
    {
      //invalid string length, one nibble must be missing
      AfxMessageBox(  _T("The data you entered is missing one nibble!\r\n") \
                      _T("Length will be adjusted!"));


      // Cut of data to valid length
      csTemp = csTemp.Left( iStrLen - 1);
      m_cOutputDataCtrl.SetWindowText(csTemp);

      m_cOutputDataCtrl.SetSel(-1, -1);
      m_cOutputDataCtrl.SetFocus();
      
    } else
    {
      int iByteCount = iStrLen / 2;

      EnterCriticalSection(&m_tcsLengthChange);

      CHANNEL_IO_INFORMATION tIOInfo = {0};
      long lRet = CcifXTestDlg::s_pcDevice->IOInfo( CIFX_IO_INPUT_AREA, 0, sizeof(tIOInfo), &tIOInfo);

      // Check only if we where able to read the size      
      if( (CIFX_NO_ERROR == lRet) &&
          (unsigned long)(iByteCount) > tIOInfo.ulTotalSize)
      {
        //packet does not fit into device mailbox
        CString csError;
        csError.Format(_T("The data you entered does not fit into the device output area!\r\n")\
                        _T("I/O area size = %u, Your data size = %u.\r\n") \
                        _T("Length will be adjusted!") ,
                        tIOInfo.ulTotalSize,
                        iByteCount);

        AfxMessageBox(csError);
        
        // Cut of data to valid length
        csTemp = csTemp.Left( tIOInfo.ulTotalSize * 2);
        m_cOutputDataCtrl.SetWindowText(csTemp);

        m_cOutputDataCtrl.SetSel(-1, -1);
        m_cOutputDataCtrl.SetFocus();

      } else
      {
      
        m_ulSendSize = (unsigned long)iByteCount;
        m_pabSendBuffer = (unsigned char*)realloc(m_pabSendBuffer, m_ulSendSize);
        
        CString csOutput;

        //parse string
        for(int iByteIdx = 0; iByteIdx < iByteCount; ++iByteIdx)
        {
          TCHAR* pszEnd = NULL;
          m_pabSendBuffer[iByteIdx] = (unsigned char)_tcstol(csTemp.Mid(iByteIdx * 2, 2), &pszEnd, 16);
          
          if((iByteIdx > 0) && ((iByteIdx % 16) == 0))
            csOutput.Append(_T("\r\n"));

          csOutput.AppendFormat(_T("%02X "), m_pabSendBuffer[iByteIdx]);
        }

        m_cOutputDataCtrl.SetWindowText(csOutput);
        iDataLen = iByteCount;
        fRet = true;
      }

      LeaveCriticalSection(&m_tcsLengthChange);
    }
  }

  TCHAR szBuffer[12] = {0};
  GetDlgItem(IDC_SEND_LENGTH)->SetWindowText( _ltot(iDataLen, szBuffer, 10));
  
  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Validate Output data
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnEnKillfocusSendData()
{
  // ValidateSendData();
}

/////////////////////////////////////////////////////////////////////////////
/// Update rate selection changed event
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnCbnSelchangeComboUpdaterate()
{
  int iCurSel = m_cUpdateRateCtrl.GetCurSel();

  ASSERT(-1 != iCurSel);

  SetThreadTimeout((unsigned long)m_cUpdateRateCtrl.GetItemData(iCurSel));
}

/////////////////////////////////////////////////////////////////////////////
/// Cyclic update checkbox clicked event
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnBnClickedChkCyclic()
{
  m_fCyclicOutput = m_cCyclicSendCtrl.GetCheck() == BST_CHECKED;

  GetDlgItem(IDC_BTN_UPDATE)->EnableWindow(!m_fCyclicOutput);
}

/////////////////////////////////////////////////////////////////////////////
/// Automatic send data increment checkbox clicked event
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnBnClickedChkAutoincr()
{
  m_fAutoIncrementData = m_cAutoIncrementCtrl.GetCheck() == BST_CHECKED;
  m_cOutputDataCtrl.EnableWindow(!m_fAutoIncrementData);
}

/////////////////////////////////////////////////////////////////////////////
/// Verify Output data checkbox clicked event
/////////////////////////////////////////////////////////////////////////////
void CIOAccessDlg::OnBnClickedChkVerifyoutput()
{
  m_fVerifyOutputs = m_cVerifyOutputsCtrl.GetCheck() == BST_CHECKED;
}
